﻿using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Net;
using System.Text;

using Newtonsoft.Json;

using VIRP.EFR.BO;

namespace VIRP.EFR.BLL
{
    public class StudyService
    {
        private AuthService AuthSvcRef;
        private AuthService AuthSvc
        {
            get
            {
                if (this.AuthSvcRef == null)
                    this.AuthSvcRef = new AuthService();

                return this.AuthSvcRef;
            }
        }

        private VistaImagingConfigurationService ConfigurationServiceRef;
        private VistaImagingConfigurationService ConfigurationService
        {
            get
            {
                if (this.ConfigurationServiceRef == null)
                    this.ConfigurationServiceRef = new VistaImagingConfigurationService();

                return this.ConfigurationServiceRef;
            }
        }

        public StudyQueryResponse GetStudiesForPatient(
            string currentUser, 
            int currentRegistryId, 
            string patientIcn)
        {
            var accessToken = this.AuthSvc.GetToken(currentUser, currentRegistryId, clearCache: true);
            if (accessToken == null)
                return null;

            var config = this.ConfigurationService.GetConfiguration(currentUser, currentRegistryId);

            var headers = AuthService.GetAuthRequestHeaders(accessToken.Value, config.SITE_NAME, config.SITE_NUMBER);

            var requestBytes = this.GetRequestAsBytes(patientIcn);

            var response = this.GetResponse(config.STUDY_QUERY_ENDPOINT, currentUser, requestBytes, headers);

            return this.ProcessResponse(response, patientIcn, currentRegistryId);
        }

        private byte[] GetRequestAsBytes(string patientIcn)
        {
            var request = new StudyQueryRequest { PatientICN = patientIcn };

            var json = JsonConvert.SerializeObject(request);

            return Encoding.UTF8.GetBytes(json);
        }

        private string GetResponse(
            string studyQueryEndpoint, 
            string currentUser, 
            byte[] requestBytes, 
            NameValueCollection headers)
        {
            string response = null;

            try
            {
                response = WebRequestor.Post(
                    studyQueryEndpoint,
                    requestBytes,
                    currentUser,
                    headers);
            }
            catch (WebException)
            {
                #region Note

                // NOTE:    As of this release VistA Imaging will throw an exception (500 error) if a patient is not found.
                //          They should return a 404.  Because this is a common occurance, we don't log the details here or 
                //          re-throw.  If the situation changes in the future, revise accordingly.
                //          HttpWebRequestExtensions.TryGetResponse will log any WebExceptions.

                #endregion
            }

            return response;
        }

        private StudyQueryResponse ProcessResponse(
            string response, 
            string patientIcn, 
            int registryId)
        {
            if (string.IsNullOrEmpty(response))
                return null;

            StudyQueryResponse validatedStudyQueryResponse = JsonSchemaValidator.DeserializeAndValidate<StudyQueryResponse>(response);
            if (validatedStudyQueryResponse != null)
                validatedStudyQueryResponse.Studies = new ImageStudies(validatedStudyQueryResponse.Studies).EncodedImageStudies;

            LogManager.LogInformation(string.Format("Found: {0} studies for patient: {1}.", validatedStudyQueryResponse.Studies.Count(), patientIcn), string.Format("{0}.{1}", System.Reflection.MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name), string.Empty, registryId);

            return validatedStudyQueryResponse;
        }

        public void InsertUpdateStudies(
            string currentUser, 
            int currentRegistryId, 
            int patientId, 
            IEnumerable<Study> studies)
        {
            var images = PATIENT_IMAGEManager.GetStagingPatientImagesFromStudies(currentUser, patientId, studies);
            if (images != null && images.Any())
                PATIENT_IMAGEManager.Save(currentUser, currentRegistryId, patientId, images);
        }
    }
}
